/* 
 * sigchld.c 
 *
 * Створює задану кількість (CHILDS_NUM) дочірніх процесів, 
 * після створення кожного витримуючи певну паузу (PARENT_TIMEOUT), 
 * невдовзі після чого завершує роботу. Кожен із дочірніх процесів
 * лише витримує певну паузу (CHILD_TIMEOUT) і завершує роботу.
 * В ході роботи перехоплює сигнали SIGCHLD і, оробляючи їх,
 * знищує дескриптори дочірніх процесів-зомбі і інкрементує
 * лічильник дочірніх процесів, що завершили роботу. Перез завершенням
 * роботи виводить кількість створених дочірніх процесів і кількість 
 * перехоплених сигналів SIGCHLD.
 * Ілюструє порядок асинхронного видалення дочірніх процесів-зомбі за
 * допомогою сигналу SIGCHLD. Застосовує функції sigaction(), wait().
 *
 */

#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>

enum { CHILDS_NUM = 10,         /* Кількість дочірніх процесів */
       CHILD_TIMEOUT = 1,
       PARENT_TIMEOUT = 1 };

volatile sig_atomic_t childs_died = 0;  /* Лічильник померлих нащадків */


/* Обробник сигналу SIGCHLD */
void cleanup_child(int signum)
{
        /* Чистить таблицю процесів (функція wait() є безпечною в
           контексті асинхронної обробки сигналів). */
        wait(NULL);
        /* Збільшує лічильник померлих. */
        childs_died++;
}

int main()
{
        struct sigaction act, oact;
        int childs_created;
        int i;

        /* Реєструє обробник сигналу SIGCHLD. */
        memset(&act, 0, sizeof(act));
        act.sa_handler = cleanup_child;
        if (sigaction(SIGCHLD, &act, &oact) != 0) {
                perror("sigaction()");
                exit(EXIT_FAILURE);
        }

        /* Робить нащадків. */
        childs_created = 0;
        for (i = 0; i < CHILDS_NUM; i++) {
                switch(fork()) {
                case -1:
                        perror("fork()");
                        break;
                case 0:
                        /* Код нащадка */
                        sleep(CHILD_TIMEOUT);
                        exit(EXIT_SUCCESS);
                default: 
                        /* Код батька */
                        childs_created++;
                        printf("Створено нащадка номер %d\n", i + 1);
                        sleep(PARENT_TIMEOUT);
                }
        }
        printf("Кількість створених нащадків: %d\n", childs_created);

        /* У цьому місці батько може виконувати будь-яку власну роботу. */
        /* ... */

        /* Прибирає обробник для SIGCHLD. */
        if (sigaction(SIGCHLD, &oact, NULL) != 0)
                perror("sigaction()");

        printf("Кількість отриманих сигналів SIGCHLD: %d\n", childs_died);

        exit(EXIT_SUCCESS);
}
